<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');
/**
* @package direct-project-innovation-initiative
* @subpackage models
*/

/** require parent */
require_once 'database_model.php';

/**
* @package direct-project-innovation-initiative
* @subpackage models
*/
class User_model extends Database_model {
	protected $primary_key = 'user_id';
	protected $_logged_in_user; //set in is_logged_in()
	protected $_is_admin;
	
	//////////////////////////////////////////
	// METHODS CONCERNING THE LOGGED IN USER
	//////////////////////////////////////////
	
	/**
	* True if a user is logged in.
	* @todo Allow a username to be passed in to check if a specific user is logged in
	* @return boolean
	*/
	public function is_logged_in(){
		if($this->session->userdata("is_loggedin") != "true") 
			return false;
		$username = $this->session->userdata('username');
		if(empty($username)) return false;
		
		//if this is the first time we've called this method since the page load, make sure that the user exists & store them for future reference.
		if(!isset($this->_logged_in_user) || $this->_logged_in_user['user_name'] != $username){
			$user = $this->db->limit(1)->get_where('users', array('user_name' => $username, 'user_deleted_flag' => 0))->row_array();
			if(empty($user) || !is_array($user)) return false;
			$this->_logged_in_user = $user;
			$this->_logged_in_user['dn'] = "uid=" . $username . "," . LDAP_ACCOUNTS_DN;
		}
		
		return true;
	}
	
	/**
	* True if the logged in user is an admin.
	* @todo Allow a username to be passed in ot check if a specific user is an admin.
	* @return boolean
	*/
	public function is_admin(){
		if(!$this->is_logged_in()) return false;
		if(!isset($this->_is_admin)){
			$this->load->library('ldap');
			$groups = $this->ldap->get_admin_group_membership($this->logged_in_user('dn'));
			$this->_is_admin = $this->is->nonzero_unsigned_integer($groups['count']);
		}
		return $this->_is_admin;
	}
	
	/**
	* True if the user user has been active within the time allowed by {@link SESSION_TIMEOUT_MINS}.
	* Additionally, updates the last activity count if the user has not been timed out.
	* @return boolean
	*/
	public function has_recent_activity(){
		//prev last activity update set to current last activity so we can reset it for functions we don't want to consider "activity"
		if(strlen($this->session->userdata('app_last_activity')) > 0) { 
			$this->session->set_userdata('prev_last_activity',$this->session->userdata('app_last_activity')); 
		}else{ //if last activity not yet set, set it to current time
			$this->session->set_userdata('app_last_activity',time()); 
		}
		
        //check if it the user needs to be logged out due to inactivity
        if(abs($this->session->userdata('app_last_activity') - time()) > (SESSION_TIMEOUT_MINS*60)){ 
			return false;
		}

		//update the last activity for the timeout
		$this->session->set_userdata('app_last_activity',time()); 
		return true;
	}
	
	/**
	* The credentials that should be used to sign into LDAP.
	* Generally, this is the logged-in user's own credentials.  If no one is logged in,
	* defaults to the credentials for the {@link LDAP_SEARCH_USERNAME} account.
	* @return array
	*/
	public function ldap_credentials(){
		$username = $this->logged_in_user('user_name');
		if(!empty($username)){
			return array( 'user' => $username,
						  'pwd' => $this->encrypt->decode($this->session->userdata('ep')));
		}
		return array('user' => LDAP_SEARCH_USERNAME, 'pwd' => LDAP_SEARCH_PASSWORD);
	}
	
	public function last_login_info() {
		return $this->db->query('SELECT TOP(2) * FROM logins WHERE username=' . $this->db->escape($this->session->userdata('username')) . ' ORDER BY login_time DESC');
	}
	
	/**
	* 
	*
	*/
	public function logged_in_user($field = null){				
		if(!$this->is_logged_in()) return array();
		if(is_null($field))
			return $this->_logged_in_user;
		if(!array_key_exists($field, $this->_logged_in_user)) return $this->error->should_be_a_recognized_user_property($field);
		return $this->_logged_in_user[$field];		
	}
	
	public function log_out($destination = 'auth'){
		$this->session->sess_destroy(); 
		session_destroy(); 
		redirect($destination);
	}
	
	////////////////////////////////////////
	// METHODS CONCERNING ALL USERS
	////////////////////////////////////////
	
	public function username_from_id($id) {
		if(!$this->is->nonzero_unsigned_integer($id)) return $this->error->should_be_a_nonzero_unsigned_int($id);
		$user = $this->db->limit(1)->get_where('users', array('user_id' => $id));
		if($user && $user->num_rows > 0) { 
			$result = $user->row_array(0);
			return $result['user_name'];
		}
		return FALSE;
	}
	
	public function user_cn_from_id($id) {
		if(!$this->is->nonzero_unsigned_integer($id)) return $this->error->should_be_a_nonzero_unsigned_int($id);
		$result = $this->ldap->search(null,1,array('cn'),'(uid='.$this->username_from_id($id).')');
		if(count($result) > 0) { return $result[0]['cn']; }
		return FALSE;
	}
	public function create_user($username,$first, $middle,$last,$department,$organization,$telephone,$mobile,$location,$title,$ext_mail,$piv_id,$edipi){
		$this->load->model('webmailmodel');
		$pass = $this->random_password();
		$userpassword = $this->ssha256_encode($pass);
		
		$db_attributes = array(
			'edipi' => (isset($edipi)) ? $edipi : null,
			'piv_id' => $piv_id,
			'username' => $username,
			'ext_mail' => $ext_mail,
			'pass' => $pass,
			'actor_id' => 0,
			'user_ext_notify_flag' => 1,
			'user_ext_group_notify_flag' => 1,
			'user_deleted_flag' => 0,
			'user_is_group' => 0,
		);
		$create_user_rec = $this->webmailmodel->create_user_record($db_attributes);
		if($create_user_rec == FALSE) { 
			return FALSE; 
		}
		
		$conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($conn, LDAP_ADMIN_USERNAME, LDAP_ADMIN_PASSWORD);	
		if($ldap_bind) {
			$attributes = array();
			$attributes["objectClass"] = array("posixAccount", "top", "person", "organizationalPerson", "inetOrgPerson");
			$attributes["gidNumber"] = "5000";
			$attributes["uidNumber"] = "5000";
			$attributes["uid"] = $username;
			$attributes["homeDirectory"] = "/var/mailboxes/" . $username;
			$attributes["cn"] =($first . " " . $last);
			$attributes["givenName"] = $first;
			$attributes["sn"] = $last;
			$attributes["mail"] = $username . "@" . DIRECT_DOMAIN;
			$attributes["userPassword"] = $userpassword;
			if(!is_null($middle) && $middle !== "") { $attributes["initials"] = $middle; $attributes["displayName"] = ($last . ", " . $first . " " . $middle); }
			else { $attributes["displayName"] = ($last . ", " . $first); }
			if(!is_null($department) && $department !== "") { $attributes["departmentNumber"] = $department; }
			if(!is_null($organization) && $organization !== "") { $attributes["o"] = $organization; }
			if(!is_null($telephone) && $telephone !== "") { $attributes["telephoneNumber"] = $telephone; }
			if(!is_null($mobile) && $mobile !== "") { $attributes["mobile"] = $mobile; }
			if(!is_null($location) && $location !== "") { $attributes["physicaldeliveryofficename"] = $location; }
			if(!is_null($title) && $title !== "") { $attributes["title"] = $title; }
			if(!ldap_add($conn, "uid=" . $attributes["uid"] . "," . LDAP_ACCOUNTS_DN,$attributes)) { 
				$this->db->query('DELETE FROM users WHERE user_name=' . $this->db->escape($username));
				return FALSE; 
			}
		}
		else {
			$this->db->query('DELETE FROM users WHERE user_name=' . $this->db->escape($username));
			return FALSE;
		}
		$get_target_id = $this->webmailmodel->get_user_id($username);
		if($get_target_id) {
			$target_id_row = $get_target_id->row_array();
			$target_id = $target_id_row["user_id"];
			$this->load->library("audit");
			$this->audit->log_event("edit",array($target_id,0,"Create User",date('U')));
			return true; 
		}
		return false;
	}
	private function random_password() {
		$chars = array("a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","1","2","3","4","5","6","7","8","9","0","!","@","#","$","%","^","&","*","(",")","{","}","?");
		$pass = "";
		for($i = 0; $i < 32; $i++) {
			$pass .= $chars[array_rand($chars)];
		}
		return $pass;
	}
	
	private function ssha256_encode($text) {
		$salt = hash('sha256', openssl_random_pseudo_bytes(32));
		$hash = "{SSHA256}".base64_encode(pack("H*",hash('sha256',$text.$salt)).$salt);
		return $hash;
	}
	private function prepare_ldap_conn() {
		$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
		if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
		if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
		return $ldap_conn;
	}
}
